<?php
declare(strict_types=1);

// Simple installer: runs migrations and seeds an admin user.

require_once __DIR__ . '/../app/lib/db.php';
require_once __DIR__ . '/../app/lib/auth.php';

$pdo = db();

$migrationsDir = __DIR__ . '/../migrations';
$files = glob($migrationsDir . '/*.sql');
sort($files);

$pdo->exec('CREATE TABLE IF NOT EXISTS app_migration (id text PRIMARY KEY, applied_at timestamptz NOT NULL DEFAULT now())');
$applied = $pdo->query('SELECT id FROM app_migration')->fetchAll(PDO::FETCH_COLUMN) ?: [];
$applied = array_flip($applied);

foreach ($files as $f) {
  $id = basename($f);
  if (isset($applied[$id])) continue;
  echo "Applying $id...\n";
  $sql = file_get_contents($f);
  if ($sql === false) throw new RuntimeException("Cannot read $f");
  $pdo->beginTransaction();
  try {
    $pdo->exec($sql);
    $stmt = $pdo->prepare('INSERT INTO app_migration(id) VALUES(:id)');
    $stmt->execute([':id'=>$id]);
    $pdo->commit();
  } catch (Throwable $e) {
    $pdo->rollBack();
    throw $e;
  }
}

echo "\nSeeding defaults...\n";

// Create default org if none
$orgId = $pdo->query('SELECT id FROM org ORDER BY created_at ASC LIMIT 1')->fetchColumn();
if (!$orgId) {
  $stmt = $pdo->prepare('INSERT INTO org(name, short_name, color) VALUES(:n,:s,:c) RETURNING id');
  $stmt->execute([':n'=>'Example Org', ':s'=>'ORG', ':c'=>'#d6002a']);
  $orgId = $stmt->fetchColumn();
}

// Create admin user if none
$adminExists = $pdo->query("SELECT 1 FROM app_user WHERE role='admin' LIMIT 1")->fetchColumn();
if (!$adminExists) {
  $hash = password_hash('admin123', PASSWORD_DEFAULT);
  $stmt = $pdo->prepare('INSERT INTO app_user(username, display_name, role, org_id, password_hash, is_active)
    VALUES(:u,:d,:r,:oid,:ph,true)');
  $stmt->execute([':u'=>'admin', ':d'=>'Admin', ':r'=>'admin', ':oid'=>$orgId, ':ph'=>$hash]);
  echo "Created admin user: admin / admin123\n";
}

// Default settings
$defaults = [
  'branding' => [
    'appName' => 'EinsatzBoard',
    'primaryColor' => '#d6002a',
    'accentColor' => '#111827',
    'logoPath' => null,
    'faviconPath' => null,
  ],
  'privacy' => [
    'pseudonymizeOtherOrgs' => true,
    'showNamesToOtherOrgs' => false,
    'retentionDays' => 180,
  ],
  'ops' => [
    'transferModeDefault' => 'offer',
    'fastModeGraceSeconds' => 120,
    'heartbeatWarnMinutes' => 30,
    'heartbeatAlarmMinutes' => 45,
  ],
  'team' => [
    'roles' => [
      ['key'=>'lead', 'label'=>'Leitung', 'sort'=>10],
      ['key'=>'nav',  'label'=>'Navigation', 'sort'=>20],
      ['key'=>'radio','label'=>'Funk', 'sort'=>30],
      ['key'=>'member','label'=>'Mitglied', 'sort'=>90],
    ],
    'allowMultipleLeads' => true,
  ],
  'orderTemplates' => [
    [
      'key' => 'search-sector',
      'label' => 'Search Sector',
      'fields' => [
        ['key'=>'area', 'label'=>'Area / Sector', 'type'=>'text', 'required'=>true],
        ['key'=>'objective', 'label'=>'Objective', 'type'=>'textarea', 'required'=>false],
        ['key'=>'notes', 'label'=>'Operational Notes', 'type'=>'textarea', 'required'=>false],
      ]
    ],
    [
      'key' => 'standby',
      'label' => 'Standby',
      'fields' => [
        ['key'=>'location', 'label'=>'Standby Location', 'type'=>'text', 'required'=>true],
        ['key'=>'until', 'label'=>'Until', 'type'=>'text', 'required'=>false],
      ]
    ]
  ],
];

foreach ($defaults as $k => $v) {
  $stmt = $pdo->prepare('INSERT INTO app_setting(key, value_json) VALUES(:k,:v::jsonb)
    ON CONFLICT (key) DO NOTHING');
  $stmt->execute([':k'=>$k, ':v'=>json_encode($v, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES)]);
}

// Seed some capability defaults (safe to ignore if you prefer to manage via Admin later)
$capSeed = [
  // global
  ['org_id'=>null, 'key'=>'hf', 'label'=>'Hundeführer', 'sort'=>10],
  ['org_id'=>null, 'key'=>'navigator', 'label'=>'Navigator', 'sort'=>20],
  ['org_id'=>null, 'key'=>'funk', 'label'=>'Funk', 'sort'=>30],
  ['org_id'=>null, 'key'=>'san', 'label'=>'Sanität', 'sort'=>40],
  ['org_id'=>null, 'key'=>'driver', 'label'=>'Fahrer', 'sort'=>50],
  ['org_id'=>null, 'key'=>'drohne', 'label'=>'Drohne', 'sort'=>60],
];

$capIns = $pdo->prepare('INSERT INTO capability(org_id, key, label, sort_order, is_active)
  VALUES(:oid, :k, :l, :s, true)
  ON CONFLICT (org_id, key) DO NOTHING');

foreach ($capSeed as $c) {
  $capIns->execute([
    ':oid' => $c['org_id'],
    ':k' => $c['key'],
    ':l' => $c['label'],
    ':s' => $c['sort'],
  ]);
}

echo "Done.\n";
